# PSCAN2: Superconductor Circuit Simulator 
# Copyright (C) 2015,2016 by Pavel Shevchenko (pscan2sim@gmail.com)

from . import psglobals
from .Expression import smooth_step
from . import SimulationParameters

class ExternalVar:
    """
    External parameter in circuit. If the whole circuit contains several 
    circuits, external parameters in each circuit will be independent

    Methods
    ---------
    Name() : string
        Name of the external parameter
    Value() : float
        Value of external parameter
    InitialValue() : float
        Initial value of external parameter (as defined in HDL file)
    SetValue(new_value)
        Set value of external parameter
    """
    def __init__(self, path, vardecl, index, circuit):
        self.set_attributes(vardecl.Attributes())
        self.path = path
        self.name = vardecl.Name()
        # circuit cannot be None
        cir_fname = circuit.FullName()
        if cir_fname == ".":
            self.full_name = "." + self.name
        else:
            self.full_name = cir_fname + "." + self.name
        self.value = vardecl.Value().GetFloatConstValue()
        self.initial_value = self.value
        self.init_smooth_time = -SimulationParameters.INITIAL_RAMP
        self.smooth_time = SimulationParameters.INITIAL_RAMP
        self.expected_value = self.value
        self.init_smooth_value = self.value
        self.index = index
        psglobals.ExternalValues[self.index] = self.value
        
    def set_attributes(self, alist):
        self.is_resettable = False
        for attr in alist:
            if attr == "resettable":
                self.is_resettable = True		

    def __repr__(self):
        return("External:[{}] {}.{} = {} [{}] ({})".format(self.is_resettable, self.path, self.name,
            self.value, self.initial_value, self.index))
            
    def Name(self):
        return(self.name)
        
    def Path(self):
        return(self.path)
        
    def FullName(self):
        return(self.full_name)
        
    def FindType(self):
        return('p')

    def Value(self):
        return(psglobals.ExternalValues[self.index])
        
    def InitialValue(self):
        return(self.initial_value)
        
    def Index(self):
        return(self.index)
        
    def IsResettable(self):
        return(self.is_resettable)    
        
    def SetValue(self, newvalue):
        self.value = newvalue
        psglobals.ExternalValues[self.index] = self.value
        
    def SetValueSmooth(self, newvalue, smooth_time = SimulationParameters.INITIAL_RAMP):
        self.init_smooth_time = psglobals.CurrentTime
        self.smooth_time = smooth_time
        self.expected_value = newvalue
        self.init_smooth_value = self.value
        
    def SetToInitialValue(self):
        self.value = self.initial_value
        psglobals.ExternalValues[self.index] = self.initial_value    
    
    def Update(self):
        if psglobals.CurrentTime >= self.init_smooth_time and \
            psglobals.CurrentTime <= self.init_smooth_time + self.smooth_time:
            self.value = self.init_smooth_value + \
                smooth_step((psglobals.CurrentTime - self.init_smooth_time)/self.smooth_time)*\
                (self.expected_value - self.init_smooth_value)
            psglobals.ExternalValues[self.index] = self.value   
        
            
